//
//  Copyright(C) 2018 OPTOELECTRONICS CO., LTD. All rights reserved.
//
/*******************************************************************************/
/** @file
 *	@filename	lib.h
 *
 * Overview:	Header file for OPH-5000i SDK library
 *
 * Description:
 *
 *------------------------------------------------------------------------------
 *	History
 *		Last Modified :
 *			@date		$Date: $
 *			@author		$Author: $
 *			@version	$Revision: $
 *******************************************************************************/

#ifndef __LIB_H__
#define __LIB_H__

//==========================================================================//
//                             Macro Definition                             //
//==========================================================================//

//---- General definitions ----//
#define OK							0
#define ERROR						-1
#define	CRITICAL					-2
#define WARNING						1
#define NG							100
#define ON							1
#define OFF							0
#define YES							1
#define NO							0
#define TRUE						1
#define FALSE						0
#define NONE						0xFF
#define DISABLED					-2
#define DISABLE						0
#define ENABLE						1
#define CURRENT_MODE				-1
#define OFF_IMMEDIATELY				0
#define ON_CONTINUOUSLY				-1

#ifndef NULL
#define NULL						0
#endif
#define HIGH						1
#define LOW							0

//---- ASCII definitions ----//
#define SOH							0x01
#define STX							0x02
#define ETX							0x03
#define EOT							0x04
#define ENQ							0x05
#define ACK							0x06
#define BEL							0x07
#define BS							0x08
#define HT							0x09
#define LF							0x0A
#define VT							0x0B
#define FF							0x0C
#define CR							0x0D
#define SO							0x0E
#define SI							0x0F
#define DLE							0x10
#define DC1							0x11
#define DC2							0x12
#define DC3							0x13
#define DC4							0x14
#define NAK							0x15
#define SYN							0x16
#define ETB							0x17
#define A_CAN						0x18
#define EM							0x19
#define SUB							0x1A
#define ESC							0x1B
#define A_FS						0x1C
#define A_GS						0x1D
#define A_RS						0x1E
#define US							0x1F
#define SPACE						0x20
#define DEL							0x7F

//---- Key definitions ----//
// Numerical Key Code
#define NUM0_KEY					'0'
#define NUM1_KEY					'1'
#define NUM2_KEY					'2'
#define NUM3_KEY					'3'
#define NUM4_KEY					'4'
#define NUM5_KEY					'5'
#define NUM6_KEY					'6'
#define NUM7_KEY					'7'
#define NUM8_KEY					'8'
#define NUM9_KEY					'9'

// Normal Key Code
#define TRIGGER_KEY					0x80
#define SCAN_KEY					0x80
#define Q1_KEY						0x81
#define Q2_KEY						0x82
#define BS_KEY						0x85
#define CLR_KEY						0x84
#define CLEAR_KEY					0x84
#define ENT_KEY						0x0A
#define SHIFT_KEY					0x8A
#define DOT_KEY						0x2E
#define F1_KEY						0x90
#define F2_KEY						0x91
#define F3_KEY						0x92
#define SIDE_KEY					0xF0

// Shift Key Code
#define LEFT_KEY					0x86
#define RIGHT_KEY					0x87
#define SHIFT_F1_KEY				0x94
#define SHIFT_F2_KEY				0x95
#define SHIFT_F3_KEY				0x96


// Key SHIFT MODE
#define ORIGINAL_SHIFT_MODE			0
#define ALTERNATE_SHIFT_MODE		1
#define MANUAL_SHIFT_MODE			2
#define ALTERNATE_SHIFT_MODE2		3
#define KINPUT_CHECK_MIN			500
#define KINPUT_CHECK_MAX			2000

//---- Display definitions ----//
#define DISPLAY_RESOLUTION_X		128
#define DISPLAY_RESOLUTION_Y		160
#define DISP_WIDTH					DISPLAY_RESOLUTION_X
#define DISP_HEIGHT					DISPLAY_RESOLUTION_Y
#define DISP_BPP					3

// Cursor
//#define OFF						0
//#define ON						1
#define BLINKING					0x2
#define AUTOWRAP					0x3
#define NOWRAP						0x4

// SetFont
#define TINY_FONT					0x01
#define SMALL_FONT					0x02
#define MEDIUM_FONT					0x03
#define LARGE_FONT					0x04
#define HUGE_FONT					0x05
#define USER_FONT					0x06
#define LARGE_FONT2					0x08
#define TINY_FONT_X_MAX				21
#define TINY_FONT_Y_MAX				26
#define SMALL_FONT_X_MAX			16
#define SMALL_FONT_Y_MAX			20
#define MEDIUM_FONT_X_MAX			21
#define MEDIUM_FONT_Y_MAX			13
#define LARGE_FONT_X_MAX			16
#define LARGE_FONT_Y_MAX			10
#define LARGE_FONT2_X_MAX			16
#define LARGE_FONT2_Y_MAX			10
#define HUGE_FONT_X_MAX				10
#define HUGE_FONT_Y_MAX				6

/* SetLineSpacing */
#define MAX_LINE_SPACING		8

// Color Definition
#define RGB_WHITE					0xFFFFFF
#define RGB_RED						0xFF0000
#define RGB_GREEN					0x00FF00
#define RGB_BLUE					0x0000FF
#define RGB_MAGENTA					0xFF00FF
#define RGB_CYAN					0x00FFFF
#define RGB_YELLOW					0xFFFF00
#define RGB_BLACK					0x000000

// BitBlt operations
#define WRITE_TO_SCREEN				0			// Just write all the pixels
#define TRANSPARENT					1			// Write all the colored pixels. The black pixels in the buffer remain unchanged on the screen
#define READ_BACK					2			// Only read the datat from the screen

// BackLightLvl
#define LCD_LEVEL1					15
#define LCD_LEVEL2					31
#define LCD_LEVEL3					63
#define LCD_LEVEL4					127
#define LCD_LEVEL5					255

// FlipScreen
#define DISP_NORMAL					0
#define DISP_REVERSE				1
//#define TOGGLE					3
//#define AUTO						4

//---- Barcode definitions ----//
// Barcode ID
#define EAN13						0x01
#define UPCA						0x02
#define EAN8						0x03
#define UPCE						0x04
//#define UPCE1						0x05		// Not support
#define ISBN						0x06
#define ISSN						0x07
#define ISMN						0x08
#define CODE39						0x09
#define CODABAR						0x0A
#define NW7							CODABAR
#define D2OF5						0x0B		// industrial 2 of 5
#define I2OF5						0x0C		// interleaved 2 of 5
#define CODE93						0x0D
#define CODE128						0x0E
#define MSI_PLESSEY					0x0F
#define IATA						0x10
#define UK_PLESSEY					0x11
#define TELEPEN						0x12
#define M2OF5						0x13
#define CHINPOST					0x14
#define NW7_ABC						0x15
#define NW7_CX						0x16
#define SCODE						0x17
#define LAETUS						0x18
#define TRIOPTIC					LAETUS
#define C39_FA						0x19		// Full ascii
#define C39_IT						0x1A		// Italian pharma code
#define EAN128						0x1B
#define RSS_14						0x1C
#define GS1_DATABAR					RSS_14
#define RSS_LIMITED					0x1D
#define RSS_EXPANDED				0x1E
#define PDF417						0x1F
#define CCC_COMPOSITE				0x20
#define MICRO_PDF417				0x21
#define CCA_COMPOSITE				0x22
#define CCB_COMPOSITE				0x23
#define CODE11						0x24
#define EAN128_CCC					0x25		// EAN-128 with CC-C composite
#define CODE3OF5					0x26		// Code 3 of 5 A.K.A. Korean Postal Code
#define JAN13						0x27 // Obsoleted in ReadBarcodeEx()
#define INSTORE13					0x28 // Obsoleted in ReadBarcodeEx()
#define JAN8						0x29 // Obsoleted in ReadBarcodeEx()
#define INSTORE8					0x2A // Obsoleted in ReadBarcodeEx()
#define INTELLIGENT_MAIL			0x2B
#define POSTNET						0x2C
#define JAPANESE_POSTAL				0x2D
#define CODABLOCK_F					0x2E
#define DATA_MATRIX_ECC200			0x2F
#define DATA_MATRIX_ECC140			0x30
#define AZTEC_CODE					0x31
#define AZTEC_RUNES					0x32
#define CHINESE_SENSIBLE_CODE		0x33
#define QR_CODE						0x34
#define MICRO_QR_CODE				0x35
#define MAXI_CODE					0x36
#define COMPOSITE_GS1DATABAR		0x37
#define COMPOSITE_UPCEAN			0x38
#define PLANET						0x39
#define NETHERLANDS_KIX				0x3A
#define UK_POSTAL					0x3B
#define AUSTRALIAN_POSTAL			0x3C
#define CHINESE_POST				0x3D
#define ADD_ON_2					0x40
#define EAN13_2						0x41
#define UPCA_2						0x42
#define EAN8_2						0x43
#define UPCE_2						0x44
#define UPCE1_2						0x45
#define CODE128_2					0x4E		// the same group CODE-128
#define ID_MAILMARK_4_STATE_POSTAL	0x4F
#define ID_PASSPORTS				0x50
#define ID_VISAS_A					0x51
#define ID_VISAS_B					0x52
#define ID_TRAVEL_DOCUMENTS_1		0x53
#define ID_TRAVEL_DOCUMENTS_2		0x54
#define ID_ISON						0x55
#define ID_JAPAN_BOOK_KIND_AND_PRICE	0x56
#define ID_TRAVEL_OCR				0x57
#define JAN13_2						0x67 // Obsoleted in ReadBarcodeEx()
#define INSTORE13_2					0x68 // Obsoleted in ReadBarcodeEx()
#define JAN8_2						0x69 // Obsoleted in ReadBarcodeEx()
#define INSTORE8_2					0x6A // Obsoleted in ReadBarcodeEx()
#define ADD_ON_5					0x80
#define EAN13_5						0x81
#define UPCA_5						0x82
#define EAN8_5						0x83
#define UPCE_5						0x84
#define UPCE1_5						0x85
#define CODE128_5					0x8E		// the same group CODE-128
#define JAN13_5						0xA7 // Obsoleted in ReadBarcodeEx()
#define INSTORE13_5					0xA8 // Obsoleted in ReadBarcodeEx()
#define JAN8_5						0xA9 // Obsoleted in ReadBarcodeEx()
#define INSTORE8_5					0xAA // Obsoleted in ReadBarcodeEx()
#define MENU_CODE					0xF0		// code39 Menu label
#define MENU_CODE_PDF				0xF1		// PDF417 Menu label
#define MENU_CODE_C128				0xF2		// Code-128 Menu label
//#define NO_READ					0xF3		// Internal code-id for a No-read message
#define MENU_CODE_NW7				0xF4		// ANSI Codabar Denso Menu label
#define MENU_CODE_QR				0xF5		// QR Menu label

//==========================================================
//  Constants for id member
//==========================================================
#define	ID_NONE						0x00		// Not Decode
#define	ID_EAN_13					0x01		// EAN-13
#define	ID_UPC_A					0x02		// UPC-A
#define	ID_EAN_8					0x03		// EAN-8
#define	ID_UPC_E					0x04		// UPC-E
//#define	ID_UPC_E1				0x05		// UPC-E1 (Not Support)
#define	ID_ISBN						0x06		// ISBN "979"
#define	ID_ISSN						0x07		// ISSN "977"
#define	ID_ISMN						0x08		// ISMN "9790"
#define	ID_CODE39					0x09		// CODE-39
#define	ID_CODABAR					0x0A		// Codabar
#define	ID_NW7						ID_CODABAR	// NW-7
#define	ID_D_2OF5					0x0B		// Industrial  2of5
#define	ID_I_2OF5					0x0C		// Interleaved 2of5
#define	ID_CODE93					0x0D		// CODE-93
#define	ID_CODE128					0x0E		// CODE-128
#define	ID_MSI_PLESSEY				0x0F		// MSI/Plessey
#define	ID_IATA						0x10		// IATA / Standard 2of5
#define	ID_UK_PLESSEY				0x11		// UK/Plessey
#define	ID_TELEPEN					0x12		// TELEPEN
#define	ID_MATRIX					0x13		// Matrix 2of5
#define	ID_MATRIX_CP				0x14		// Chinese post Matrix 2of5
#define	ID_NW7_ABC					0x15		// Codabar ABC
#define	ID_NW7_CX					0x16		// Codabar CX
#define	ID_SCODE					0x17		// Scode
#define	ID_LAETUS					0x18		// Code39 Laetus
#define	ID_TRIOPTIC					ID_LAETUS	// Trioptic / Laetus
#define	ID_C39_FUL					0x19		// Code39 Full ASCII
#define	ID_C39_ITP					0x1A		// Code39 Itarian Pharmaceutical
#define	ID_EAN128					0x1B		// EAN-128
#define	ID_RSS_14					0x1C		// RSS 14
#define	ID_GS1_DATABAR				ID_RSS_14	// GS1 DataBar / RSS 14
#define	ID_RSS_LIMITED				0x1D		// RSS Limited
#define	ID_RSS_EXPANDED				0x1E		// RSS Expanded
#define	ID_PDF417					0x1F		// PDF417
#define	ID_CCC_COMPOSITE			0x20		// Commposite C (PDF417)
#define	ID_MICRO_PDF417				0x21		// MicroPDF417
#define	ID_CCA_COMPOSITE			0x22		// Commposite A (MicroPDF417)
#define	ID_CCB_COMPOSITE			0x23		// Commposite B (MicroPDF417)
#define	ID_CODE11					0x24		// CODE-11
#define	ID_EAN128_CCC				0x25		// EAN-128 with CC-C composite
#define	ID_C_3OF5					0x26		// Code 3of5
#define	ID_JAN_13					0x27		// JAN-13
#define	ID_INST_13					0x28		// Instore-13
#define	ID_JAN_8					0x29		// JAN-8
#define	ID_INST_8					0x2A		// Instore-8

#define ID_INTELLIGENT_MAIL			0x2B
#define ID_POSTNET					0x2C
#define ID_JAPANESE_POSTAL			0x2D
#define ID_CODABLOCK_F				0x2E
#define ID_DATA_MATRIX_ECC200		0x2F
#define ID_DATA_MATRIX_ECC140		0x30
#define ID_AZTEC_CODE				0x31
#define ID_AZTEC_RUNES				0x32
#define ID_CHINESE_SENSIBLE_CODE	0x33
#define ID_QR_CODE					0x34
#define ID_MICRO_QR_CODE			0x35
#define ID_MAXI_CODE				0x36
#define ID_COMPOSITE_GS1DATABAR		0x37
#define ID_COMPOSITE_UPCEAN			0x38
#define ID_PLANET					0x39
#define ID_NETHERLANDS_KIX			0x3A
#define ID_UK_POSTAL				0x3B
#define ID_AUSTRALIAN_POSTAL		0x3C

#define ID_MAILMARK_4_STATE_POSTAL	0x4F

#define ID_PASSPORTS				0x50
#define ID_VISAS_A					0x51
#define ID_VISAS_B					0x52
#define ID_TRAVEL_DOCUMENTS_1		0x53
#define ID_TRAVEL_DOCUMENTS_2		0x54
#define ID_ISON						0x55
#define ID_JAPAN_BOOK_KIND_AND_PRICE	0x56


// ScannerPower mode
//#define OFF						0
//#define ON						1
#define TRIGGER						2
#define SINGLE						3
#define MULTIPLE					4

//---- Communication definitions ----//
// Comm port number
#define COM3						3			//  Bluetooth SPP/Master
#define COM4						4			//  Bluetooth HID/Slave
#define COM5						5			//  Bluetooth SPP/Slave
#define COM9						9			//  USBCOM
#define COM10						10			//  USBHID
#define COM13						13			//  Bluetooth HID/Master (Not Support)

// Bluetooth setting definitions
#define BLT_BD_ADDR_LEN				12			// BD address length
#define BLT_MAX_PINCODE_LEN			16			// Max PIN Code length
#define BLT_CLASS_OF_DEVICE_LEN		6			// BD Class of device
#define BLT_MAX_LOCAL_NAME_LEN		30			// Max device name length
#define BLT_MAX_REMOTE_NAME_LEN		30			// 
#define BLT_MAX_PASSKEY_LEN			9			// Max Passkey length

// Communication Error
#define COM_ERROR_00				(0)			// Not Error
#define COM_ERROR_01				(-1)		// Parameter Error
#define COM_ERROR_02				(-2)		// COM Number Not Supported
#define COM_ERROR_03				(-3)		// Max SIO Num Over
#define COM_ERROR_04				(-4)		// Device Not Find
#define COM_ERROR_05				(-5)		// Connect Failed
#define COM_ERROR_06				(-6)		// Send Error
#define COM_ERROR_07				(-7)		// Receive TimeOut
#define COM_ERROR_08				(-8)		// Status Error
#define COM_ERROR_09				(-9)		// Exclusive Device Select
#define COM_ERROR_10				(-10)		// CONFIGURATION Error
#define COM_ERROR_13				(-13)		// Cancel Event
#define COM_ERROR_14                (-14)       // Receive Error

// GetAuthenticationStatus
#define COM_NOT_AUTHENTICATED		1			// NOT AUTHENTICATED
#define COM_PIN_REQUIRED			2			// PIN REQUIRED
#define COM_AUTHENTICATED			3			// AUTHENTICATED

// GetConnectionStatus
#define COM_STATE_CLOSE				1			// CLOSE
#define COM_STATE_CONNECT			2			// CONNECT
#define COM_STATE_DISCONNECT		3			// DISCONNECT
#define COM_STATE_WAIT_ACCEPT		6			// WAIT ACCEPT (bluetooth only)
#define COM_STATE_ACCEPTED			8			// ACCEPTED	 (bluetooth only)
//#define ERROR						-1

// Cradle mode
#define CRD_NO_USE					0			// Default
#define CRD_USE_RS232C				1			// Cradle RS-232C
#define CRD_USE_HID					2			// Cradle HID

// HID Output code to Numeric data
#define HID_OUTPUTCODE_FULL			0x00		// Full Key code      (Default)
#define HID_OUTPUTCODE_NUMPAD		0x01		// Numpad code

// HID Caps Lock Mode
#define HID_CAPS_MODE_AUTO1			0x00		// Auto 1
#define HID_CAPS_MODE_NOCAPS		0x01		// No Caps Lock Mode  (Default)
#define HID_CAPS_MODE_CAPS			0x02		// Caps Lock Mode
#define HID_CAPS_MODE_AUTO2			0x03		// Auto 2 (Always Capslock change before output data)

// HID Num Lock Mode
#define HID_NUM_MODE_AUTO			0x00		// Auto               (Default)
#define HID_NUM_MODE_NONUM			0x01		// No Num Lock Mode
#define HID_NUM_MODE_NUM			0x02		// Num Lock Mode

// HID Output Release Capslock code for German and French keyboard
#define HID_GR_FR_KB_DIS			0x00		// Disable
#define HID_GR_FR_KB_ENA			0x01		// Enable             (Default)

// SPP Inter Character Delay Time
#define BLT_SPP_INTER_DELAY1_NO		0			// No Delay           (Default)
#define BLT_SPP_INTER_DELAY1_20		1			// Delay = 20ms
#define BLT_SPP_INTER_DELAY1_50		2			// Delay = 50ms , actually 2.5
#define BLT_SPP_INTER_DELAY1_100	5			// Delay = 100ms

// HID Inter Character Delay Time
#define BLT_INTER_DELAY1_NO			0			// No Delay           (Default)
#define BLT_INTER_DELAY1_1			10			// Delay = 10ms
#define BLT_INTER_DELAY1_2			20			// Delay = 20ms
#define BLT_INTER_DELAY1_3			30			// Delay = 30ms
#define BLT_INTER_DELAY1_4			40			// Delay = 40ms
#define BLT_INTER_DELAY1_5			50			// Delay = 50ms
#define BLT_INTER_DELAY1_6			60			// Delay = 60ms
#define BLT_INTER_DELAY1_7			70			// Delay = 70ms
#define BLT_INTER_DELAY1_8			80			// Delay = 80ms
#define BLT_INTER_DELAY1_9			90			// Delay = 90ms
#define BLT_INTER_DELAY1_10			100			// Delay = 100ms
#define BLT_INTER_DELAY1_11			150			// Delay = 150ms
#define BLT_INTER_DELAY1_12			200			// Delay = 200ms
#define BLT_INTER_DELAY1_13			250			// Delay = 250ms
#define BLT_INTER_DELAY1_14			300			// Delay = 300ms
#define BLT_INTER_DELAY1_15			350			// Delay = 350ms
#define BLT_INTER_DELAY1_16			400			// Delay = 400ms
#define BLT_INTER_DELAY1_17			450			// Delay = 450ms
#define BLT_INTER_DELAY1_18			500			// Delay = 500ms
#define BLT_INTER_DELAY1_19			550			// Delay = 550ms
#define BLT_INTER_DELAY1_20			600			// Delay = 650ms

// OseComm
#define ANY_KEYS_ABORT				0x0200
#define CRADLE_ABORT				0x0100
#define SUCC_COMPLETE				1

// OseComm Return States
#define COMM_OK						(0)			// Communication OK
#define ERR_PR_NO_RESPONSE			(-1)		// No response
#define ERR_PR_CRC					(-2)		// CRC-16 error
#define ERR_PR_CMD_MISMATCH			(-3)		// Wrong command received
#define ERR_PR_FRAME_LENGTH			(-4)		// Frame length error
#define ERR_PR_SESSION_ID			(-5)		// Session ID mismatch
#define ERR_PR_SENDING_FRAME		(-6)		// Error sending frame
#define ERR_PR_OPEN_FILE			(-7)		// Cannot open or create the requested file
#define ERR_PR_READ_FROM_FILE		(-8)		// Cannot read from file
#define ERR_PR_WRITE_TO_FILE		(-9)		// cannot write to file
#define ERR_PR_WRONG_BLOCK			(-10)		// Requested the wrong block number
#define ERR_PR_FILE_SIZE			(-11)		// Error in file sizes
#define ERR_PR_POLL_COMMAND			(-12)		// unexpected poll command
#define ERR_PR_FILE_UNAVAILABLE		(-13)		// File cannot be found in the terminal
#define ERR_PR_LINE_BUSY			(-14)		// An other terminal is already communicating
#define ERR_PR_TIME_DATE			(-15)		// Error in the time or date when it is being set
#define ERR_PR_DELETE_FILE			(-16)		// Error deleting file
#define ERR_PR_USER_ABORT			(-20)		// User aborted the protocol.

// OseComm STATUS that are used in the callback (OnOseCommInfo)
#define STAT_GET_TIME_DATE			1
#define STAT_SET_TIME_DATE			2
#define STAT_GET_OS_VERSION			3
#define STAT_GET_APPL_VERSION		4
#define STAT_XMIT_FILE_TO_PC		5
#define STAT_RECV_FILE_FROM_PC		6
#define STAT_LIST_FILES				7
#define STAT_DELETE_FILES			8

// MFi
enum{
	iAP2_EASs_NONE = 0,							// Initialize state
	iAP2_EASs_CONNECT,							// External accessory session is connected
	iAP2_EASs_DISCONNECT						// External accessory session is disconnected
};

//---- Power definitions ----//
// AutoPowerDown
#define AP_RESET					0x02
#define AP_NOW						0x03
#define AP_RESUME					0x04
#define AP_SUSPEND					0x05
#define APD_OFF						OFF			// To stpo the auto powwer down sunction
#define APD_ON						ON			// To start the auto power down function
#define APD_RESET					AP_RESET	// Reset the time to auto power down
#define APD_NOW						AP_NOW		// Power off immediately
#define APD_ENABLE					AP_RESUME	// Restart the auto power down function
#define APD_DISABLE					AP_SUSPEND	// Suspend the auto power down function
// Wake up key
enum WAKEUP_KEY
{
	WAKEUP_ANYKEY	=	0,
	WAKEUP_POWERKEY
};
// Power on key
enum POWERON_KEY
{
	POWERON_ANYKEY	=	0,
	POWERON_POWERKEY
};

// IsChargingState
#define OFF_CRADLE					0
#define CHARGING					1
#define ON_CRADLE					2
#define CHARGING_DISABLED			3
#define CHARGE_PENDING				4
#define CHARGING_USB				5
#define CHARGING_CRADLE				6
#define COMPLETED_USB				7
#define COMPLETED_CRADLE			8

//---- Time definitions ----//
#define BASE_TIME					20
#define TIME_1SEC					(1000/BASE_TIME)	// 1sec
#define TIME_2SEC					(2000/BASE_TIME)	// 2sec
#define TIME_5SEC					(5000/BASE_TIME)	// 5sec
#define TIME_10SEC					(10000/BASE_TIME)	// 10sec
#define TIME_100MS					5

//---- I/O definitions ----//
// GoodReadLed
#define RED							0x01
#define GREEN						0x02
#define BLUE						0x04
#define ORANGE						(RED|GREEN)
#define RED_FLASH					0x10
#define GREEN_FLASH					0x20
#define BLUE_FLASH					0x40
#define ORANGE_FLASH				(RED_FLASH|GREEN_FLASH)
#define RED_QFLASH					0x100
#define GREEN_QFLASH				0x200
#define BLUE_QFLASH					0x400
#define ORANGE_QFLASH				(RED_QFLASH|GREEN_QFLASH)

// Sound Time
#define TCLICK						1
#define TSHORT						3
#define TSTANDARD					5
#define TLONG						10
#define TVLONG						20

// Sound Volume
#define VLOW						1
#define VSTANDARD					8
#define VMEDIUM						32
#define VHIGH						127

// Sound Tone
#define SERROR						100
#define SLOW						1000
#define SMEDIUM						2500
#define SHIGH						3500
#define SPAUSE						1

// Key Click Sound
#define KC_LOW						4
#define KC_STANDARD					8
#define KC_MEDIUM					32
#define KC_HIGH						127

// Key BackLight
#define KEYBL_TIME_OFF				0xFF
#define KEYBL_TIME_2SEC				2
#define KEYBL_TIME_5SEC				5
#define KEYBL_TIME_10SEC			10
#define KEYBL_TIME_ON				0
#define RESET_APD					2
#define TOGGLE						3
#define AUTO						4
#define NOW							5
#define SUSPEND						6
#define RESUME_APD					7
#define RETURN_TIME					8

#define MIN_BL_VAL					15
#define LOW_BL						MIN_BL_VAL

// Vibrate Power Level
#define VIB_POWER_OFF				0
#define VIB_POWER_MIN				1
#define VIB_POWER_MID				2
#define VIB_POWER_MAX				3
#define VIB_POWER_PAUSE 			100

//---- File and Memory definitions ----//
#define D_NAME						8
#define D_EXT						3
#define MAX_PATH					260

// fcopy mode
#define WRITE_ONLY					0
#define WRITE_VERIFY				1

// *pFileCopy mode
#define COPY_FILE					0
#define VERIFY_FILE					1

// Return value for fcopy
#define FCOPY_NO_MEM				-3
#define FCOPY_ILLEGAL_PRM			-4
#define DUPLICATE_FILE				-5
#define UNKNOWN_SRC_FILE			-6
#define ZERO_SRC_FILE				-7
#define WRONG_DEST_FILE				-8
#define FAIL_SRC_FILE				-9
#define FAIL_DEST_FILE				-10
#define VERIFY_ERROR				-11

//---- Special definitions ----//
// RunApplFromFile
#define RAP_ERR_FILETYPE			-1
#define RAP_ERR_FILENAME			-4
#define RAP_ERR_PARAM				-5
#define USER_APP1					2
#define USER_APP2					3

// SetUsbMsdMode
#define MSD_ABORT_KEY				0xFF

//==========================================================================//
//                           WLAN definitions                               //
//==========================================================================//

// Security Type
#define M2M_WIFI_SEC_OPEN				1
#define M2M_WIFI_SEC_WPA_PSK			2
#define M2M_WIFI_SEC_WEP				3
#define M2M_WIFI_SEC_802_1X				4

// Channel
#define M2M_WIFI_CH_1					1
#define M2M_WIFI_CH_2					2
#define M2M_WIFI_CH_3					3
#define M2M_WIFI_CH_4					4
#define M2M_WIFI_CH_5					5
#define M2M_WIFI_CH_6					6
#define M2M_WIFI_CH_7					7
#define M2M_WIFI_CH_8					8
#define M2M_WIFI_CH_9					9
#define M2M_WIFI_CH_10					10
#define M2M_WIFI_CH_11					11
#define M2M_WIFI_CH_12					12
#define M2M_WIFI_CH_13					13
#define M2M_WIFI_CH_ALL					255

// Socket
#define SOCKET_BUFFER_MAX_LENGTH		1400
#define AF_INET							2
#define SOCK_STREAM						1
#define SOCK_DGRAM						2
#define SOCKET_FLAGS_SSL				1

// Socket Callback
#define SOCKET_MSG_BIND					1
#define SOCKET_MSG_LISTEN				2
#define SOCKET_MSG_DNS_RESOLVE			3
#define SOCKET_MSG_ACCEPT				4
#define SOCKET_MSG_CONNECT				5
#define SOCKET_MSG_RECV					6
#define SOCKET_MSG_SEND					7
#define SOCKET_MSG_SENDTO				8
#define SOCKET_MSG_RECVFROM				9

// WLAN Callback
#define M2M_WIFI_RESP_CONN_INFO			6
#define M2M_WIFI_REQ_CONNECT			40
#define M2M_WIFI_REQ_DEFAULT_CONNECT	41
#define M2M_WIFI_RESP_DEFAULT_CONNECT	42
#define M2M_WIFI_REQ_DISCONNECT			43
#define M2M_WIFI_RESP_CON_STATE_CHANGED	44
#define M2M_WIFI_REQ_SLEEP				45
#define M2M_WIFI_REQ_WPS_SCAN			46
#define M2M_WIFI_REQ_WPS				47
#define M2M_WIFI_REQ_START_WPS			48
#define M2M_WIFI_REQ_DISABLE_WPS		49
#define M2M_WIFI_REQ_DHCP_CONF			50
#define M2M_WIFI_RESP_IP_CONFIGURED		51
#define M2M_WIFI_RESP_IP_CONFLICT		52
#define M2M_WIFI_REQ_ENABLE_MONITORING	53
#define M2M_WIFI_REQ_DISABLE_MONITORING	54
#define M2M_WIFI_RESP_WIFI_RX_PACKET	55
#define M2M_WIFI_REQ_SEND_WIFI_PACKET	56
#define M2M_WIFI_REQ_LSN_INT			57
#define M2M_WIFI_REQ_DOZE				58
#define M2M_WIFI_MAX_STA_ALL			59

#define M2M_MAC_ADDRES_LEN								6
/*!< The size fo 802 MAC address.
 */
#define M2M_MAX_SSID_LEN 								33
/*!< 1 more than the max SSID length.
	This matches the size of SSID buffers (max SSID length + 1-byte length field).
 */
#define M2M_MAX_PSK_LEN           						65
/*!< 1 more than the WPA PSK length (in ASCII format).
	This matches the size of the WPA PSK/Passphrase buffer (max ASCII contents + 1-byte length field).
	Alternatively it matches the WPA PSK length (in ASCII format) + 1 byte NULL termination.
 */
#define M2M_MIN_PSK_LEN           						9
/*!< 1 more than the minimum WPA PSK Passphrase length.
	It matches the minimum WPA PSK Passphrase length + 1 byte NULL termination.
 */
#define M2M_AUTH_1X_USER_LEN_MAX                        100
/*!< The maximum length (in ASCII characters) of domain name + username (including '@' or '\')
	for authentication with Enterprise methods.
*/
#define M2M_AUTH_1X_USER_OLD_LEN_SUB					32
/*!< The difference length (in ASCII characters) of domain name + username (including '@' or '\')
	for old authentication with Enterprise methods.
*/
#define M2M_AUTH_1X_PASSWORD_LEN_MAX					256
/*!< The maximum length (in ASCII characters) of password for authentication with Enterprise MSCHAPv2 methods.
*/
#define M2M_AUTH_1X_PRIVATEKEY_LEN_MAX					256
/*!< The maximum length (in bytes) of private key modulus for authentication with Enterprise TLS methods.
	Private key exponent must be the same length as modulus, pre-padded with 0s if necessary.
*/
#define M2M_AUTH_1X_CERT_LEN_MAX						1584
/*!< The maximum length (in bytes) of certificate for authentication with Enterprise TLS methods.
*/

// Connect State
#define M2M_WIFI_DISCONNECTED			0
#define M2M_WIFI_CONNECTED				1
#define M2M_WIFI_UNDEF					0xff
// Connect error state
typedef enum {
	 M2M_ERR_SCAN_FAIL = ((unsigned char)1),
	/*!< Indicate that the WLAN board has failed to perform the scan operation.
	*/
	 M2M_ERR_JOIN_FAIL,
	/*!< Indicate that the WLAN board has failed to join the BSS .
	*/
	 M2M_ERR_AUTH_FAIL,
	/*!< Indicate that the WLAN board has failed to authenticate with the AP.
	*/
	 M2M_ERR_ASSOC_FAIL,
	/*!< Indicate that the WLAN board has failed to associate with the AP.
	*/
	 M2M_ERR_CONN_INPROGRESS,
	 /*!< Indicate that the WLAN board has another connection request in progress.
	*/
	 M2M_ERR_DISC_RESET = 10
	 /*!< Indicate that the WLAN board has reset.
	*/
}tenuM2mConnChangedErrcode;

// (legacy)tenuPowerSaveModes(SetPowerSaveModeWlan)
#define M2M_NO_PS						0
#define M2M_PS_AUTOMATIC				1
#define M2M_PS_H_AUTOMATIC				2
#define M2M_PS_DEEP_AUTOMATIC			3
#define M2M_PS_MANUAL					4
// SetPowerSaveModeWlan/SysSetWLANConfigValue
#define WLAN_PS_DISABLE					M2M_NO_PS
#define WLAN_PS_ENABLE					M2M_PS_DEEP_AUTOMATIC

// etc
#define M2M_SUCCESS						0
#define M2M_ERR_SLEEP_FAIL				-20
#define TCP_SOCK_MAX					7
#define UDP_SOCK_MAX					4
#define CONF_SW_TIMER_COUNT				1
#define MAX_SOCKET						(TCP_SOCK_MAX + UDP_SOCK_MAX)
#define SOL_SOCKET						1

// Socket option
#define SOL_SSL_SOCKET					2		// SSL Socket option level.
#define	SO_SET_UDP_SEND_CALLBACK		0x00	// Socket option used by the application to enable/disable the use of UDP send callbacks.
#define SO_SSL_BYPASS_X509_VERIF		0x01	// Allow an opened SSL socket to bypass the X509 certificate verification process.

#define _htons(A)	(unsigned short)((((unsigned short) (A)) << 8) | \
                    (((unsigned short) (A)) >> 8))
#define _htonl(m)	(unsigned long)(((unsigned long)(m << 24)) | \
                    ((unsigned long)((m & 0x0000FF00) << 8)) | \
                    ((unsigned long)((m & 0x00FF0000) >> 8)) | \
                    ((unsigned long)(m >> 24)))
#define _ntohl		_htonl
#define _ntohs		_htons

// Enable/Disable (SysSetWlanPower)
#define SYS_WLAN_POWER_MANUAL			0
#define SYS_WLAN_POWER_OFF				1
#define SYS_WLAN_POWER_AUTO				2

// WLAN status (SysGetWLANConnectStatus)
#define SYS_WLAN_STATUS_UNAVAILABLE		0
#define SYS_WLAN_STATUS_DISCONNECTED	1
#define SYS_WLAN_STATUS_CONNECTED		2

// WLAN config ID
#define SYS_WLAN_CONFIG_ID_REGULATORY			1
#define SYS_WLAN_CONFIG_ID_ROAMINGMODE			5
#define SYS_WLAN_CONFIG_ID_ROAMINGTHRESHOLD		6
#define SYS_WLAN_CONFIG_ID_DHCP					7
#define SYS_WLAN_CONFIG_ID_IP_ADDRESS			8
#define SYS_WLAN_CONFIG_ID_GATEWAY				9
#define SYS_WLAN_CONFIG_ID_DNS					10
#define SYS_WLAN_CONFIG_ID_SUBNETMASK			11
#define SYS_WLAN_CONFIG_ID_APINFOINDEX			20
#define SYS_WLAN_CONFIG_ID_POWERSAVEMODE		21
#define SYS_WLAN_CONFIG_ID_DNS2					22

//[id=SYS_WLAN_CONFIG_ID_REGULATORY]
#define NORTH_AMERICA					((unsigned short) 0x7FF)	// 11 channels
#define EUROPE							((unsigned short) 0x1FFF)	// 13 channels
//#define ASIA							((unsigned short) 0x3FFF)	// 14 channels (Not support)

// [id=SYS_WLAN_CONFIG_ID_ROAMINGMODE]
#define WLAN_ROAMING_NORMAL				0
#define WLAN_ROAMING_ANY				1
#define WLAN_ROAMING_STEALTH			2

// [id=SYS_WLAN_CONFIG_ID_ROAMINGTHRESHOLD]
// -100 ~ 0

// [id=SYS_WLAN_CONFIG_ID_DHCP]
#define WLAN_CONFIG_DHCP_DISABLE		0
#define WLAN_CONFIG_DHCP_ENABLE			1


#define WLAN_SOCKUDP_SEND_DISABLE		0
#define WLAN_SOCKUDP_SEND_ENABLE		1

#define SOCK_STREAM						1
#define SOCK_DGRAM						2

#define SYS_WLAN_CONFIG_ID_SSID			0
#define SYS_WLAN_CONFIG_ID_PASSWORD		1

#define WLAN_CONFIG_LENGTH_SSID			33
#define WLAN_CONFIG_LENGTH_PASSWORD		65

#define M2M_MAX_SSID_LEN 				33
#define M2M_1X_USR_NAME_MAX				21						// The maximum size of the user name including the NULL termination.
#define M2M_1X_PWD_MAX					41						// The maximum size of the password including the NULL termination.
#define WEP_40_KEY_STRING_SIZE 			((unsigned char)10)		// Indicate the wep key size in bytes for 40 bit string passphrase.
#define WEP_104_KEY_STRING_SIZE 		((unsigned char)26)		// Indicate the wep key size in bytes for 104 bit string passphrase.

#define AP_ROAMING_MODE_ANY_INCLUSION	0
#define AP_ROAMING_MODE_ANY_EXCLUSION	1

// Socket Errors
#define SOCK_ERR_NO_ERROR				0		// Successful socket operation.
#define SOCK_ERR_INVALID_ADDRESS		-1		// Socket address is invalid.
#define SOCK_ERR_ADDR_ALREADY_IN_USE	-2		// Socket operation cannot bind on the given address.
#define SOCK_ERR_MAX_TCP_SOCK			-3		// Exceeded the maximum number of TCP sockets.
#define SOCK_ERR_MAX_UDP_SOCK			-4		// Exceeded the maximum number of UDP sockets.
#define SOCK_ERR_INVALID_ARG			-6		// An invalid argument is passed to a function.
#define SOCK_ERR_MAX_LISTEN_SOCK		-7		// Exceeded the maximum number of TCP passive listening sockets.
#define SOCK_ERR_INVALID				-9		// The requested socket operation is not valid in the current socket state.
#define SOCK_ERR_ADDR_IS_REQUIRED		-11		// Destination address is required.
#define SOCK_ERR_CONN_ABORTED			-12		// The socket is closed by the peer. The local socket is closed also.
#define SOCK_ERR_TIMEOUT				-13		// The socket pending operation has timedout.
#define SOCK_ERR_BUFFER_FULL			-14		// No buffer space available to be used for the requested socket operation.
#define SOCK_ERROR						-100	// Fatal error. Wlan module to restart.
#define SOCK_CLOSE						-101	// Remote peer socket is closed.
#define SOCK_RECV_ERROR					-102	// Receive fail.
#define SOCK_FAIL						-103	// Fail.


#define SSID_MODE_VISIBLE				0		// SSID is visible to others.
#define SSID_MODE_HIDDEN				1		// SSID is hidden.

#define PING_ERR_SUCCESS				0
#define PING_ERR_DEST_UNREACH			1
#define PING_ERR_TIMEOUT				2
#define PING_ERR_FAIL					-1
#define PING_ERR_WRONG_RESPONSE			-2

enum WLAN_CERT_MODE
{
	WLAN_CERT_SET = 0,							// set data
	WLAN_CERT_CLEAR,							// clear data
};


//==========================================================================//
//                           Structure Definition                           //
//==========================================================================//

// Barcode structure
struct barcode
{
	char *text;
	int length;
	int id;
	int min;
	int max;
	char opticonID;
	char *aimID;
};
struct barcodeEx
{
	char *text;
	int length;
	int id;
	int min;
	int max;
	char opticonID;
	char aimID[4];
};

// Charging structure
typedef union
{
	unsigned short WORD;
	struct {
		unsigned char FAULT:4;
		unsigned char STAT:2;
		unsigned char V50:1;
		unsigned char CE:1;
		unsigned char BATTMP:1;
		unsigned char SCAN:1;
		unsigned char SYSOFF:1;
		unsigned char INIT:1;
		unsigned char TIMEOUT:1;
		unsigned char CRADLE:1;
		unsigned char BATTMP2:2;
	} BIT;
} CHG_STS;

// Date structure
struct date_struct
{
	unsigned short da_year;
	unsigned char da_day;
	unsigned char da_mon;
};

// Time structure
struct time_struct
{
	unsigned char ti_hour;
	unsigned char ti_min;
	unsigned char ti_sec;
};

// File Block structure
struct ffblk
{
	char name[D_NAME+1];
	char ext[D_EXT+1];
	int filesize;
	char search_name[D_NAME+1];
	char search_ext[D_EXT+1];
	int slot;
	int drive;
	unsigned char attrib;
};
/* File attribute bits for attrib */
#define	AM_RDO	0x01	/* Read only */
#define	AM_HID	0x02	/* Hidden */
#define	AM_SYS	0x04	/* System */
#define	AM_VOL	0x08	/* Volume label */
#define AM_LFN	0x0F	/* LFN entry */
#define AM_DIR	0x10	/* Directory */
#define AM_ARC	0x20	/* Archive */
#define AM_MASK	0x3F	/* Mask of defined bits */

typedef enum
{
	IO_CAP_DISPLAY_ONLY = 0,			// Display Only.
	IO_CAP_DISPLAY_YES_NO = 1,			// Display Yes/No.
	IO_CAP_KEYBOARD_ONLY = 2,			// Keyboard Only.
	IO_CAP_NO_INPUT_NO_OUTPUT = 3,		// No IO.
	IO_REJECT_REQUES = 4				// Use this to reject the IO capability request
} IO_CAPABILITY_REQUEST;

enum BLT_AUTHENTICATION
{
	AUTH_DISABLE_AUTD = 0,
	AUTH_ENABLE_AUTE = 1,
	AUTH_ENABLE_AUTO = 7
};

enum IPHONE_MODE
{
	DISABLE_IPHONE_MODE = 0,
	ENABLE_IPHONE_MODE_SPP_MASTER,
	ENABLE_IPHONE_MODE_SPP_SLAVE,
	ENABLE_IPHONE_MODE_HID
};

enum CHARACTER_CODE_KIND
{
	NOT_USE_MULTI_BYTE = 0xFF,
	WINDOWS_MULTI_BYTE = 0,
	SHIFT_JIS_MULTI_BYTE,
	GB18030_MULTI_BYTE,
	BIG5_MULTI_BYTE,
	UHC_MULTI_BYTE,
	UTF8_MULTI_BYTE,
	UTF16_MULTI_BYTE,
	UTF16LE_MULTI_BYTE,
	UTF16BE_MULTI_BYTE,
	IBM_MULTI_BYTE
};

enum UPPER_ASCII_OUT_MODE
{
	OUTMODE_NORMAL = 0,
	OUTMODE_ONLY_LOW_ASCII,
	OUTMODE_NOT_OUT_KANJI,
	OUTMODE_KANJI_MODE_MOST,
	OUTMODE_KANJI_MODE_NUMPAD
};

enum MULTI_BYTE_TRANSMIT_MODE
{
	OUTMODE_NORMAL_ASCII = 0,
	OUTMODE_MULTI_BYTE
};

enum	TransmitWait
{
	COMLIBLO_WAIT,
	NOT_WAIT,
	USE_TX_ISR
};

// Serial Config Structure
typedef volatile struct
{
	unsigned int		baud_rate;				// Serial output speed
	unsigned int		data_bit;				// Data bit
	unsigned int		parity_bit;				// Parity bit
	unsigned int		stop_bit;				// Stop bit
	unsigned int		handshaking;			// Handshaking
	unsigned int		delay_time;				// Inter Character Delay Time
	unsigned int		cts_wait;				// Extend Flow Control Time (CTS Wait Time)
	unsigned int		rts_ctl;				// RTS Control
	enum TransmitWait	transmitWait;			//
} setconfig_serial;

// HID Config Structure
typedef volatile struct
{
	unsigned int		output_code;			// Output code to Numeric data
	unsigned int		kb_language;			// Keyboard Language
	unsigned int		inter_delay;			// Inter Character Delay Time
	unsigned int		caps_mode;				// Caps Lock Mode
	unsigned int		num_mode;				// Num Lock Mode
	unsigned int		gr_fr_kb;				// Output Release Capslock code for German and French keyboard
	unsigned short		userVendorID;
	unsigned short		userProductID;
	unsigned int		buffered_mode;			// Buffering out mode
	unsigned char		maxPower;
	unsigned char		intervalTime;
	unsigned char		reserved;
} setconfig_hid;

// Ir Config Structure
typedef volatile struct
{
	unsigned int		ir_power;				// Ir Power Control
} setconfig_ir;

// USBHID Config Structure
typedef volatile struct
{
	unsigned int		emulation;				// Emulation Mode
} setconfig_usbhid;

// Wedge Config Structure
typedef volatile struct
{
	unsigned int		kb_layout;				// Keyboard layout
	unsigned int		keyboard;				// Keyboard
	unsigned int		jp_hira;				// Japanese Hiragana Mode
	unsigned int		jp_kata;				// Japanese Katakana Mode
	unsigned int		inter_delay2;			// Inter Character Delay Time 2
	unsigned int		scan_code;				// Scan code set
} setconfig_wdg;

// RF Config Structure
#define MAX_PARTNER_NO		1					// Register Cradle Number
typedef volatile struct
{
	//--------------------------------
	union
	{
		unsigned char	addrB[8];
		unsigned long	addrDW[2];
	} reg_addr[MAX_PARTNER_NO] __attribute__ ((packed));		// 8byte
	unsigned int	rf_channel __attribute__ ((packed));		// 4byte
	//--------------------------------
	unsigned char	rf_local_addr[8];							// 8byte
	//--------------------------------
} setconfig_zigbee;												// 20byte

// Bluetooth Config Structure
typedef volatile struct
{
	char address[BLT_BD_ADDR_LEN+1];							// BD Address                    12+1Byte
	char pincode[BLT_MAX_PINCODE_LEN+1];						// PIN Code                      16+1byte
	char classOfDevice[BLT_CLASS_OF_DEVICE_LEN+1];				// Class of device                6+1byte
	unsigned int encryption;									// Encryption (Enable/Disable)      4byte
	enum BLT_AUTHENTICATION  authentication;					// Authentication (Enable/Disable/AUTO)  4byte
	
	unsigned short sppInterCharDelay;							// Character Delay Unit 1ms 2byte
	unsigned short hidInterCharDelay;							// Character Delay Unit 1ms 2byte
	unsigned char language;										// keyboard language  1byte
	unsigned char dynamicPIN;									// Valiable PIN flag (Enable/Disable)   1byte
	enum IPHONE_MODE iPhoneMode;								// 1byte
	unsigned char erasePairedDevice;							// Erase paried device 1Byte
	unsigned char pairedDeviceConnect;							// Pairing device information under Slave connect
	IO_CAPABILITY_REQUEST ioCapability;							//
	enum CHARACTER_CODE_KIND characterCode;						//
	enum UPPER_ASCII_OUT_MODE upperAsciiOutMode;				//
	char altCharacterHeader[1];									//
	unsigned char cradleMode;									// CRD-3301
	char passkey[BLT_MAX_PASSKEY_LEN+1];						//
	enum MULTI_BYTE_TRANSMIT_MODE hidMultiByteTransmitMode;		// Multi byte Transmit mode      1byte
	unsigned int keyboardLfOutput;								// LF Code Output (Enable/Disable)    4byte
	
	unsigned int output_code;									// Output code to Numeric data
	unsigned int caps_mode;										// Caps Lock Mode
	unsigned int num_mode;										// Num Lock Mode
	unsigned int gr_fr_kb;										// Output Release Capslock code for German and French keyboard
} setconfig_bluetooth;

// USBCOM Config Structure
enum USBCOM_TYPE
{
	OPTO_VCP,
	OPTO_CDC
};

enum USB_RECIEVE_TYPE
{
	USB_RECIEVE_ONE_TIME,
	USB_RECIEVE_CONTINUITY,
};

typedef volatile struct
{
	unsigned short userVendorID;
	unsigned short userProductID;
	enum USBCOM_TYPE usbType;
	enum USB_RECIEVE_TYPE receiveType;
	unsigned char delayCodeWaitMode;
	unsigned char maxPower;
	unsigned char detectDsr;
} setconfig_usbcom;

// Cradle Config Structure
typedef volatile struct
{
	// RS-232C
	unsigned int		baud_rate;				// Serial output speed
	unsigned int		data_bit;				// Data bit
	unsigned int		parity_bit;				// Parity bit
	unsigned int		stop_bit;				// Stop bit
	unsigned int		handshaking;			// Handshaking
	unsigned int		delay_time;				// Inter Character Delay Time
	unsigned int		cts_wait;				// Extend Flow Control Time (CTS Wait Time)
	unsigned int		rts_ctl;				// RTS Control
	unsigned int		serialAckNakProtocol;	// ACK/NACK  CRD-3301 is not support

	// HID
	unsigned int		output_code;			// Output code to Numeric data
	unsigned int		kb_language;			// Keyboard language
	unsigned int		inter_delay;			// Inter Character Delay Time
	unsigned int		caps_mode;				// Caps Lock Mode
	unsigned int		num_mode;				// Num Lock Mode
	unsigned int		gr_fr_kb;				// Output Release Capslock code for German and French keyboard
	unsigned char		intervalTime;

	// Keyword
	unsigned char		productname[16+1];		// keyword of config
} setconfig_cradle;

// COMAPI Config Structure
typedef volatile struct
{
	setconfig_serial	serial;					// COM Common Settings
	setconfig_hid		hid;					// HID Common Settings
	setconfig_ir		ir;						// Ir Settings
	setconfig_usbhid	usbhid;					// USBHID Settings
	setconfig_wdg		wdg;					// Wedge Settings
	setconfig_zigbee	zb;						// Zigbee Settings
	setconfig_bluetooth	blt;					// Bluetooth Settings
	setconfig_usbcom	usbcom;					// USBCOM settings
	setconfig_cradle	cradle;					// Cradle
} setconfig_comapi;

// Discover Bluetooth device structure
typedef struct{
    char bd_address[BLT_BD_ADDR_LEN+1];
	char active;
	int rssi;
	int rssi_raw;
	unsigned int last_detected;
} BLT_DISCOVER_DEVICE_INFO;

typedef struct{
	int count;
	BLT_DISCOVER_DEVICE_INFO *dev_info;
} BLT_DISCOVER_RESULT;

// WLAN setting structure
typedef signed char		SOCKET;

typedef struct
{
	signed char		status;
} tstrSocketBindMsg;

typedef struct
{
	signed char		status;
} tstrSocketListenMsg;

typedef struct
{
	unsigned long		s_addr;
} in_addr;

struct sockaddr_in
{
	unsigned short		sin_family;
	unsigned short		sin_port;
	in_addr				sin_addr;
	unsigned char		sin_zero[8];
};

typedef struct
{
	SOCKET				sock;
	struct sockaddr_in	strAddr;
} tstrSocketAcceptMsg;

typedef struct
{
	unsigned char		*pu8Buffer;
	short				s16BufferSize;
	unsigned short		u16RemainingSize;
	struct sockaddr_in	strRemoteAddr;
} tstrSocketRecvMsg;

typedef struct
{
	unsigned char		u8FrameType;
	unsigned char		u8FrameSubtype;
	unsigned char		u8ServiceClass;
	unsigned char		u8Priority;
	unsigned char		u8HeaderLength;
	unsigned char		u8CipherType;
	unsigned char		au8SrcMacAddress[6];
	unsigned char		au8DstMacAddress[6];
	unsigned char		au8BSSID[6];
	unsigned short		u16DataLength;
	unsigned short		u16FrameLength;
	unsigned long		u32DataRateKbps;
	char				s8RSSI;
	unsigned char		__PAD24__[3];
} tstrM2MWifiRxPacketInfo;

struct sw_timer_module;
typedef void (*tpfAppWifiCb) (unsigned char u8MsgType, void *pvMsg);
typedef void (*tpfAppEthCb) (unsigned char u8MsgType, void *pvMsg, void *pvCtrlBuf);
typedef void (*tpfAppMonCb) (tstrM2MWifiRxPacketInfo *pstrWifiRxPacket, unsigned char *pu8Payload, unsigned short u16PayloadSize);
typedef void (*tpfAppSocketCb) (SOCKET sock, unsigned char u8Msg, void *pvMsg);
typedef void (*tpfAppResolveCb) (unsigned char * pu8DomainName, unsigned long u32ServerIP);
typedef void (*tpfAppMonCb) (tstrM2MWifiRxPacketInfo *pstrWifiRxPacket, unsigned char * pu8Payload, unsigned short u16PayloadSize);
typedef void (*sw_timer_callback_t)(struct sw_timer_module *const module, int timer_id, void *context, int period);

struct sockaddr
{
	unsigned short sa_family;
	unsigned char  sa_data[14];
};

struct sw_timer_handle {
	unsigned char used:1;
	unsigned char callback_enable:1;
	unsigned char busy:1;
	sw_timer_callback_t callback;
	void *context;
	unsigned long period;
	unsigned long expire_time;
};

struct sw_timer_module {
	struct sw_timer_handle handler[CONF_SW_TIMER_COUNT];
	unsigned long accuracy;
};

typedef struct
{
	tpfAppWifiCb pfAppWifiCb;
	tpfAppEthCb  pfAppEthCb;
	unsigned char *au8ethRcvBuf;
	unsigned short u16ethRcvBufSize;
} tstrEthInitParam;

typedef struct
{
	tpfAppWifiCb pfAppWifiCb;
	tpfAppMonCb  pfAppMonCb;
	tstrEthInitParam strEthInitParam;
} tstrWifiInitParam;

typedef struct
{
	unsigned char u8CurrState;
	unsigned char u8ErrCode;
	unsigned char __PAD16__[2];
} tstrM2mWifiStateChanged;

typedef void (*sw_timer_callback_t)(struct sw_timer_module *const module, int timer_id, void *context, int period);

typedef struct{
	SOCKET sock;
	signed char s8Error;
} tstrSocketConnectMsg;

typedef void (*print_char_function)(unsigned char data);

struct device_info
{
	char bd_address[13];						// discovered Bluetooth(R) address
	char name[16];								// discovered device name, could be empty
};

// Callback function prototype
typedef void (*pOnOseCommInfo)( int status, int errorsuccess, int progress, const char *info );

typedef struct {
	unsigned char	au8SSID[33];
	unsigned char	u8ListenChannel;
	unsigned char	u8KeyIndx;
	unsigned char	u8KeySz;
	unsigned char	au8WepKey[26 + 1];
	unsigned char	u8SecType;
	unsigned char	u8SsidHide;
	unsigned char	au8DHCPServerIP[4];
	unsigned char	__PAD24__[3];
} tstrM2MAPConfig;

typedef struct{
	char			acSSID[M2M_MAX_SSID_LEN];	// AP connection SSID name
	unsigned char	u8SecType;					// Security type
	unsigned char	au8IPAddr[4];				// Connection IP address
	unsigned char	au8MACAddress[6];			// MAC address of the peer Wi-Fi station
	signed char		s8RSSI;						// Connection RSSI signal
	unsigned char	u8CurrChannel;				// RF channel number  1,2,...
	unsigned char	__PAD__[2];					// Padding bytes for forcing 4-byte alignment
} tstrM2MConnInfo;

// WEP security key parameters.
typedef struct{
	unsigned char	u8KeyIndx;					// Wep key Index.
	unsigned char	u8KeySz;					// Wep key Size. It include Null terminated string.(WepKey length + 1)
	unsigned char	au8WepKey[WEP_104_KEY_STRING_SIZE + 1];		// WEP Key represented as a NULL terminated ASCII string.
	unsigned char	__PAD24__[3];				// Padding bytes to keep the structure word alligned.
} tstrM2mWifiWepParams;


// Credentials for the user to authenticate with the AAA server (WPA-Enterprise Mode IEEE802.1x).
typedef struct{
	unsigned char	au8UserName[M2M_1X_USR_NAME_MAX];	// User Name. It must be Null terminated string.
	unsigned char	au8Passwd[M2M_1X_PWD_MAX];			// Password corresponding to the user name. It must be Null terminated string.
} tstr1xAuthCredentials;

// WLAN connection parameters.
typedef struct{
	union{
		unsigned char			au8PSK[WLAN_CONFIG_LENGTH_PASSWORD];	// Pre-Shared Key in case of WPA-Personal security.
		tstr1xAuthCredentials	strCred1x;								// Credentials for RADIUS server authentication in case of WPA-Enterprise security.
		tstrM2mWifiWepParams	strWepInfo;								// WEP key parameters in case of WEP security.
	} key;																// Union holding all possible authentication parameters corresponding the current security types.
	unsigned char				u8SecType;								// Wi-Fi network security type. See tenuM2mSecType for supported security types.
	unsigned char				u8RoamingModeAny;						// For roaming mode any 0:inclusion 1:exclusion
	unsigned char				__PAD__[1];
	unsigned short				u16Ch;									// RF Channel for the target SSID.
	unsigned char				au8SSID[M2M_MAX_SSID_LEN];				// SSID of the desired AP. It must be NULL terminated string.
	unsigned char 				reserve;
} tstrWlanApInfo;

// WPA/PSK
typedef struct {
	unsigned char   au8Passphrase[M2M_MAX_PSK_LEN];
	/*!< Pointer to Passphrase (Printable ASCII).
			Must be NULL if PSK is provided instead. */
	unsigned char   u8PassphraseLen;
	/*!< Length of Passphrase. Permitted values are between 8 and 63.
			This field is ignored if pu8Passphrase == NULL. */
	unsigned char				isPskHexstr;		// 1
}tstrPsk;
// WEP
typedef struct {
	unsigned char   au8WepKey[WEP_104_KEY_STRING_SIZE + 1];
	/*!< Pointer to WEP Key, represented as an ASCII string.
			(10 or 26 characters, representing 5 or 13 bytes.) */
	unsigned char   u8KeySz;
	/*!< Size of WEP Key string.
			Permitted values are @ref WEP_40_KEY_STRING_SIZE or @ref WEP_104_KEY_STRING_SIZE. */
	unsigned char   u8KeyIndx;
	/*!< WEP Key Index in the range 1 to 4. */
}tstrWep;
// PEAP/EAP-TTLS
typedef struct {
	unsigned char   au8Domain[M2M_AUTH_1X_USER_LEN_MAX + 1];
	/*!< Pointer to Domain of authentication server (printable ASCII), including '@' or '\'
			separator character as appropriate. Use NULL if there is no domain information.
			The Domain will be either prepended or appended to the UserName, depending on the
			setting of field bPrependDomain. \n
			Example 1: if [Domain]is "@my_domain" and bPrependDomain is false, then the EAP
			identity response is "[UserName]@my_domain". \n
			Example 2: if [Domain]is "my_domain\" and bPrependDomain is true, then the EAP
			identity response is "my_domain\[UserName]". */
	unsigned char   reserve1[M2M_AUTH_1X_USER_OLD_LEN_SUB];
	/*!< Adjustment bytes. Before FW19.6.1, M2M_AUTH_1X_USER_LEN_MAX was 132.>*/
	unsigned char   au8UserName[M2M_AUTH_1X_USER_LEN_MAX + 1];
	/*!< Pointer to UserName (ASCII).
			This will be sent (encrypted) in the tunneled EAP identity response (if applicable)
			and used during MSCHAPv2 authentication. If bUnencryptedUserName is true then it will
			also be sent (unencrypted) in the initial EAP identity response. */
	unsigned char   reserve2[M2M_AUTH_1X_USER_OLD_LEN_SUB];
	/*!< Adjustment bytes. Before FW19.6.1, M2M_AUTH_1X_USER_LEN_MAX was 132.>*/
	unsigned char   au8Password[M2M_AUTH_1X_PASSWORD_LEN_MAX + 1];
	/*!< Pointer to MSCHAPv2 Password (ASCII).
			This will be used during MSCHAPv2 authentication. */
	unsigned short  u16DomainLen;
	/*!< Length of Domain (in ASCII characters), including '@' or '\' separator character as
			appropriate.
			Permitted values are such that u16DomainLen + u16UserNameLen is between 0 and
			@ref M2M_AUTH_1X_USER_LEN_MAX. */
	unsigned short  u16UserNameLen;
	/*!< Length of UserName (in ASCII characters).
			Permitted values are such that u16DomainLen + u16UserNameLen is between 0 and
			@ref M2M_AUTH_1X_USER_LEN_MAX. */
	unsigned short  u16PasswordLen;
	/*!< Length of Password (in ASCII characters).
			Permitted values are between 0 and @ref M2M_AUTH_1X_PASSWORD_LEN_MAX. */
	int	bUnencryptedUserName;
	/*!< Determines whether UserName or "anonymous" is sent (unencrypted) in the initial EAP
			identity response. Domain is sent in both cases. \n
			true: UserName is sent in the initial EAP identity response (not recommended).
			false: "anonymous" is sent in the initial EAP identity response. This setting is
			recommended for tunneled methods. MSCHAPv2 is always a tunneled method. */
	int	bPrependDomain;
	/*!< Determines whether Domain is prepended or appended to UserName in EAP identity responses.
			true: Domain is prepended to UserName - [Domain][UserName].
			false: Domain is appended to UserName - [UserName][Domain]. */
}tstrMschap2;
// EAP-TLS
typedef struct {
	unsigned char   au8Domain[M2M_AUTH_1X_USER_LEN_MAX + 1];
	/*!< Pointer to Domain of authentication server (printable ASCII), including '@' or '\'
			separator character as appropriate. Use NULL if there is no domain information.
			The Domain will be either prepended or appended to the UserName, depending on the
			setting of field bPrependDomain. \n
			Example 1: if [Domain]is "@my_domain" and bPrependDomain is false, then the EAP
			identity response is "[UserName]@my_domain". \n
			Example 2: if [Domain]is "my_domain\" and bPrependDomain is true, then the EAP
			identity response is "my_domain\[UserName]". */
	unsigned char   reserve1[M2M_AUTH_1X_USER_OLD_LEN_SUB];
	/*!< Adjustment bytes. Before FW19.6.1, M2M_AUTH_1X_USER_LEN_MAX was 132.>*/
	unsigned char   au8UserName[M2M_AUTH_1X_USER_LEN_MAX + 1];
	/*!< Pointer to UserName (ASCII).
			This will be sent (encrypted) in the tunneled EAP identity response.
			If bUnencryptedUserName is true then it will also be sent (unencrypted) in the initial
			EAP identity response. */
	unsigned char   reserve2[M2M_AUTH_1X_USER_OLD_LEN_SUB];
	/*!< Adjustment bytes. Before FW19.6.1, M2M_AUTH_1X_USER_LEN_MAX was 132.>*/
	unsigned short  u16DomainLen;
	/*!< Length of Domain (in ASCII characters), including '@' or '\' separator character as
			appropriate.
			Permitted values are such that u16DomainLen + u16UserNameLen is between 0 and
			@ref M2M_AUTH_1X_USER_LEN_MAX. */
	unsigned short  u16UserNameLen;
	/*!< Length of UserName (in ASCII characters).
			Permitted values are such that u16DomainLen + u16UserNameLen is between 0 and
			@ref M2M_AUTH_1X_USER_LEN_MAX. */
	int	bUnencryptedUserName;
	/*!< Determines whether UserName or "anonymous" is sent (unencrypted) in the initial EAP
			identity response. Domain is sent in both cases. \n
			true: UserName is sent in the initial EAP identity response (required for EAP-TLS).
			false: "anonymous" is sent in the initial EAP identity response. This setting is
			recommended for tunneled methods such as EAP-PEAP/TLS. */
	int	bPrependDomain;
	/*!< Determines whether Domain is prepended or appended to UserName in EAP identity responses.
			true: Domain is prepended to UserName - [Domain][UserName].
			false: Domain is appended to UserName - [UserName][Domain]. */
	unsigned char	clientcertIndex;	// 1 EAP-TLS
}tstrEapTls;
// AP setting (add EAP)
typedef struct {
	unsigned char	u8SecType;						// 1
	unsigned char	au8SSID[M2M_MAX_SSID_LEN];		// 33
	unsigned char	au8Bssid[M2M_MAC_ADDRES_LEN];	// 6
	unsigned char	u8Options;						// 1
	//tenuCredStoreOption	enuCredStoreOption;	// 1
	unsigned char	u8RoamingModeAny;				// 1 roaming
	unsigned char	eapType;						// 1
	unsigned char	isPskHexstr;					// 1
	unsigned char	reserve[84];					// 84
	union {
		unsigned char		au8PSK[M2M_MAX_PSK_LEN];	//tstrPsk				psk;
		tstrWep				strWepInfo;			//
		tstrEapTls			eapTls;				//
		tstrMschap2			Mschap2;			//
	}key;
}tstrWlanApInfoEap;

// Static IP configuration.
typedef struct{
	unsigned int		u32StaticIP;		// The static IP assigned to the device.
	unsigned int		u32Gateway;			// IP of the Default internet gateway.
	unsigned int		u32DNS;				// IP for the DNS server.
	unsigned int		u32SubnetMask;		// Subnet mask for the local area network.
	unsigned int		u32DhcpLeaseTime;	// Dhcp Lease Time in sec
	unsigned int		u32AlternateDNS;	// IP for the secondary DNS server (if any). Must set to zero if not provided in static IP configuration from the application.
} tstrM2MIPConfig;

// SysSetWLANConfigValue
typedef union{
	unsigned int		regulatory;
	unsigned int		dhcp;
	unsigned int		staticIP;
	unsigned int		gateway;
	unsigned int		dns;
	unsigned int		dns2;
	unsigned int		subnetMask;
	unsigned int		roamingmode;
	int					roamingThreshold;
	unsigned int		apinfoIndex;
	unsigned int		powerSaveMode;
} tuWlanConfigValue;

// GetSystemInfo()
typedef struct {
	char SystemVersion[16];
	char BootVersion[16];
	char OsVersion[16];
	char BatchVersion[16];
	char DecoderVersion[16];
	char WlanVersion[16];
	char BluetoothVersion[16];
	char WlanMac[18];
	char BluetoothAddress[18];
} SystemInfo;

//==========================================================================//
//                            Function Prototype                            //
//==========================================================================//
//---- Key Library ----//
unsigned int kbhit(void);
void ResetKey(void);
int ShiftState(int mode);
int IsTriggerPressed(void);
int IsUpPressed(void);
int IsDownPressed(void);
int IsSideKeyPressed(void);
int SetUseKeyClick(int mode);
int SetUseKeyClick2(int mode);
int SetSideKey(int mode);
int SetSideKeyVibrator(int mode);
unsigned short SetKeyinputTimer(unsigned short time);
short SetKeyinputMode(short mode);
int SetSideKeyCode(int code);
int IsEachkeyPressed(char key);
int IsMultiplekeyPressed(char key1, char key2, char key3);

//---- Display Library ----//
void Cursor(int mode);
int SetFont(unsigned int type, char *name);
void GetFont(unsigned int *type, char **font_ptr);
void GetFontSize(unsigned int *height, unsigned int *width);
int SetLineSpacing(int lineSpacing);
int GetLineHeight(void);
void GotoXY(int x, int y);
int WhereX(void);
int WhereY(void);
int SaveDisplay(char *filename);
int LoadDisplay(char *filename);
int SetEcho(int echomode);
void DrawLine(int x1, int y1, int x2, int y2, int color);
void SetPixel(int x, int y, int color);
void PrintSymbol(unsigned int symbol);
void ShowPicture(int x_offs, int y_offs, const unsigned char *ptr);
int FlipScreen(int mode);
void SetOffset(int offset);
int GetOffset(void);
long SetTextColor(long rgb);
long SetBackColor(long rgb);
long SetCursorColor(long rgb);
void ShowBMP(int x_offs, int y_offs, unsigned short mode, unsigned char *bmp);
int ShowJPG_File(int x_offs, int y_offs, unsigned short mode, char *filename);
int ShowJPG(int x_offs, int y_offs, unsigned short mode, unsigned char *picture);
int ShowGIF(unsigned short x_offs, unsigned short y_offs, unsigned short mode, unsigned char *picture);
int BackLightLvl(int bl_level);
void ClearDisplay(void);
void ScrollUp(unsigned short height);
void FillScreen(unsigned short start_x, unsigned short start_y, unsigned short end_x, unsigned short end_y, unsigned long color);
void DrawCircle(int cx, int cy, int r, int color);
void FillCircle(int cx, int cy, int r, int color);
void DrawEllipse(int cx, int cy, int xradius, int yradius, int color);
void FillEllipse(int cx, int cy, int xradius, int yradius, int color);
int OsStatus(int mode);
int GetScreen(int x, int y, int width, int height, unsigned char *buffer);
int SetScreen(int x, int y, int width, int height, unsigned char *buffer);

//---- Barcode Library ----//
signed int ReadBarcode(struct barcode *barcodep);
signed int ReadBarcodeEx(struct barcodeEx *barcodep);
int IsScannerOff(void);
int ScannerPower(int mode, int time);

//---- Communication Library ----//
int ComOpen(int port);
void ComClose(int port);
int comloc(void);
int GetCom(int timeout);
int PutCom(unsigned char data);
int putnstr(const unsigned char *string, int len);
int PutString(const char *string);
void ResetReceive(void);
int GetConfig(const unsigned int com_port, void *setting );
int SetConfig(const unsigned int com_port, void *setting );
int LoadConfig(const unsigned int com_port);
int SaveConfig(const unsigned int com_port);
int SetDefault(const unsigned int com_port);
char* BltGetConnectAddress(void);
void BltSetConnectAddress(const char *address);
char* BltGetPincode(void);
int BltSetPincode(const char *pincode);
char* BltGetLocalName(void);
int BltSetLocalName(const char *localname);
int BltGetLocalAddress(char *BD_ADDR);
int blt_is_connected(void);
int BltWaitAcceptStart(void);
int BltWaitAcceptEnd(int accept);
int GetConnectionStatus(const unsigned int com_port);
int GetCurrentPort(void);
int GetErrorStatus(void);
long OseComm(long sessionID, int terminateKey, const char *appVersion, pOnOseCommInfo OnOseCommInfo);
int BltStartDiscovery(int max_count);
BLT_DISCOVER_RESULT *BltFetchDiscovery(void);
void BltEndDiscovery(void);
int AccessoryDataSend(unsigned char *string, unsigned int length);
int AccessoryDataTransfer(unsigned char *string, unsigned int sendLength);
int AccessoryDataReceive(unsigned char *buffer);
int GetMfiAccessoryStatus(void);


//---- Power Library ----//
void AutoPowerDown(int what, unsigned long time);
unsigned long GetAPDTime(void);
void Idle(void);
void Wait(unsigned int time);
unsigned int GetBatteryVoltage(void);
int IsChargingState(void);
int ChargeIndicator(int mode);
int IsPoweredDown(void);
void dipoweroff(unsigned short sw);
void PowerOff(void);
void SoftReset(void);
unsigned char SetWakeupKey(unsigned char mode);
unsigned char SetPoweronKey(unsigned char mode);

//---- Realtime Clock Library ----//
short CheckDate(struct date_struct *datep);
short CheckTime(struct time_struct *timep);
short GetDate(struct date_struct *datep);
short GetTime(struct time_struct *timep);
short SetDate(struct date_struct *datep);
short SetTime(struct time_struct *timep);

//---- Timer Library ----//
void Delay(int time);
void StartTimer(unsigned int timevalue);
int EndTimer(void);
unsigned int TimerValue(void);
void SetSystemTime(unsigned long time);
unsigned long GetSystemTime(void);
unsigned int GetTickCount(void);

//---- File Library ----//
unsigned long CoreLeft(void);
int Format(void);
int findfirst(const char *filename, struct ffblk *ffblkp);
int findnext(struct ffblk *ffblkp);
long fsize(char *filename);
typedef void (*pFileCopy)(unsigned char stat, unsigned char progress);
int fdatetime(char *filename, struct date_struct *datep, struct time_struct *timep);
int fcopy(char *filename1, char *filename2, int mode, pFileCopy FileCopyCb);
int f_mkdir(const char *path);
int f_chdir(const char *path);
int f_getcwd (char *buff, unsigned int len);

//---- I/O Library ----//
void GoodReadLed(int on_off, int time);
void Sound(int time, int vol, ...);
int Backlight(int what, unsigned short time);
int KeyBacklightTime(int time);
void Vibrate(int time, int power);

//---- System Setting Library ----//
unsigned int GetTerminalId(void);
unsigned int SetTerminalId(unsigned int id);
int SystemSetting(const char *option_string);
unsigned int UseResume(int mode);
void VersionOs(char *os);
void VersionRom(char *rom);
void GetModelNo(char *modelno);
void GetSerialNo(char *serialno);
void SaveSettings(void);
unsigned int UseScreenshot(int mode);
void GetSystemInfo(SystemInfo *info);
int isSjisSupported(void);
int SetScanner(const char *menuStr);

//---- NFC Library ----//
int IsNFCConnect(struct date_struct *stDate, struct time_struct *stTime);
int ReadNFCTag(unsigned short Index, unsigned short Count, unsigned char *pBuf);
int WriteNFCTag(unsigned short Index, unsigned short Count, unsigned char *pBuf);

//---- Special Library ----//
int RunApplFromFile(char *filename);
int RunApplFromFile2(char *filename);
int GetCurrentApplMode(void);
int SetUsbMsdMode(int key);

//---- WLAN Library ----//
signed char InitializeWlan(tstrWifiInitParam * param);
signed char HandleEventWlan(struct sw_timer_module *swt_module_inst);
signed char ConnectWlan(char *pcSsid, unsigned char u8SsidLen, unsigned char u8SecType, void *pvAuthInfo, unsigned short u16Ch);
signed char DisconnectWlan(void);
signed char GetConnectInfoWlan(void);
signed char InitBsp(void);
void InitSc(void);
void RegSocketCallbackSc(tpfAppSocketCb pfAppSocketCb, tpfAppResolveCb pfAppResolveCb);
SOCKET SocketSc(unsigned short u16Domain, unsigned char u8Type, unsigned char u8Flags);
signed char BindSc(SOCKET sock, struct sockaddr *pstrAddr, unsigned char u8AddrLen);
signed char ListenSc(SOCKET sock, unsigned char backlog);
signed char AcceptSc(SOCKET sock, struct sockaddr *addr, unsigned char *addrlen);
signed char ConnectSc(SOCKET sock, struct sockaddr *pstrAddr, unsigned char u8AddrLen);
short RecvSc(SOCKET sock, void *pvRecvBuf, unsigned short u16BufLen, unsigned long u32Timeoutmsec);
short RecvfromSc(SOCKET sock, void *pvRecvBuf, unsigned short u16BufLen, unsigned long u32Timeoutmsec);
short SendSc(SOCKET sock, void *pvSendBuffer, unsigned short u16SendLength, unsigned short flags);
short SendtoSc(SOCKET sock, void *pvSendBuffer, unsigned short u16SendLength, unsigned short flags, struct sockaddr *pstrDestAddr, unsigned char u8AddrLen);
signed char CloseSc(SOCKET sock);
signed char EnableAp(tstrM2MAPConfig *apconfig);
signed char DisableAp(void);
int SetPowerSaveModeWlan(unsigned char sleepMode);
int SetSockUdpSendResponse(SOCKET sock, int response);
int setsockopt(SOCKET sock, unsigned char u8Level, unsigned char option_name, const void *option_value, unsigned short u16OptionLen);
int SetStaticIPWlan(void);
int gethostbyname(unsigned char *pcHostName);
unsigned int nmi_inet_addr(char *pcIpAddr);

//---- WLAN Auto connect AP ----//
int SysWLANInit(void *rsv);
int SysSetWLANPower(int powerStatus);
int SysGetWLANPower(int *powerStatus);
int SysGetWLANConnectStatus(int *status);
int SysGetWLANSignalLevel(int *level);
int SysGetConnectInfoWlan(tstrM2MConnInfo *info);
int SysSetWLANConfigValue(int id, tuWlanConfigValue value);
int SysGetWLANConfigValue(int id, tuWlanConfigValue *value);
int SysSetWLANApInfo(int index, tstrWlanApInfo *apInfo);
int SysGetWLANApInfo(int index, tstrWlanApInfo *apInfo);
int SysSetWLANApInfoEap(int index, tstrWlanApInfoEap *apInfo);
int SysGetWLANApInfoEap(int index, tstrWlanApInfoEap *apInfo);
int SysGetDhcpInfoWlan(tstrM2MIPConfig *dhcpInfo);
int SysSetWLANConfigString(int id, int length, char *str);
int SysGetWLANConfigString(int id, int bufferSize, char *str);
int SysSetWLANRootcert(int certIndex, char *rootcertPath, int mode);
int SysSetWLANClientcert(int certIndex, char *clientcertPath, char *privatekeyPath, int mode);

// WLAN Synchronized Socket Library ----//
void SyncInitSc(void);
SOCKET SyncSocketSc(unsigned char u8Type, unsigned char u8Flags);
int SyncBindSc(SOCKET sock, struct sockaddr_in *pstrAddr);
int SyncListenSc(SOCKET sock);
int SyncAcceptSc(SOCKET *sock, struct sockaddr_in *addr);
int SyncConnectSc(SOCKET sock, struct sockaddr_in *pstrAddr, unsigned long u32Timeoutmsec);
int SyncRecvSc(SOCKET sock, void *pvRecvBuf, unsigned short u16BufLen, unsigned long u32Timeoutmsec, unsigned short *u16RemainingLen);
int SyncRecvfromSc(SOCKET sock, void *pvRecvBuf, unsigned short u16BufLen, unsigned long u32Timeoutmsec, unsigned short *u16RemainingLen);
int SyncRecvfromAddrSc(SOCKET sock, void *pvRecvBuf, unsigned short u16BufLen, unsigned long u32Timeoutmsec, unsigned short *u16RemainingLen, struct sockaddr_in *pstrRemoteAddr);
int SyncSendSc(SOCKET sock, void *pvSendBuffer, unsigned short u16SendLength);
int SyncSendtoSc(SOCKET sock, void *pvSendBuffer, unsigned short u16SendLength, struct sockaddr_in *pstrDestAddr);
int SyncCloseSc(SOCKET sock);
int SyncGetIp(unsigned char * hostName, SOCKET sock, unsigned int *ipAddress);
int SyncPing(unsigned int ipAddress, unsigned char ttl, int *rtt);// Test API

#endif // __LIB_H__
